<?php

class FileConnector {
    
    private $base_path;
    
    public function __construct() {
        $this->base_path = __DIR__ . '/files/';
        
        if (!is_dir($this->base_path)) {
            mkdir($this->base_path, 0755, true);
        }
    }
    
    private function getFilePath($resource) {
        $file_path = $this->base_path . $resource;
        
        if (!file_exists($file_path)) {
            throw new Exception("File not found: " . $resource);
        }
        
        return $file_path;
    }
    
    public function get($params, $resource) {
        $file_path = $this->getFilePath($resource);
        $extension = pathinfo($file_path, PATHINFO_EXTENSION);
        
        switch (strtolower($extension)) {
            case 'json':
                return $this->getFromJson($params, $file_path);
            case 'csv':
                return $this->getFromCsv($params, $file_path);
            case 'txt':
                return $this->getFromText($params, $file_path);
            default:
                return file_get_contents($file_path);
        }
    }
    
    public function add($params, $resource) {
        $file_path = $this->base_path . $resource;
        $extension = pathinfo($file_path, PATHINFO_EXTENSION);
        
        switch (strtolower($extension)) {
            case 'csv':
                return $this->addToCsv($params, $file_path);
            case 'txt':
                return $this->addToText($params, $file_path);
            default:
                return file_put_contents($file_path, $params, FILE_APPEND);
        }
    }
    
    public function update($params, $resource) {
        $file_path = $this->getFilePath($resource);
        $extension = pathinfo($file_path, PATHINFO_EXTENSION);
        
        switch (strtolower($extension)) {
            case 'json':
                return $this->updateJson($params, $file_path);
            default:
                throw new Exception("Update not supported for this file type");
        }
    }
    
    public function search($params, $resource) {
        $file_path = $this->getFilePath($resource);
        $content = file_get_contents($file_path);
        
        if (preg_match("/text\s+['\"]([^'\"]+)['\"]/i", $params, $matches)) {
            $search_term = $matches[1];
            $lines = explode("\n", $content);
            $results = [];
            
            foreach ($lines as $line_num => $line) {
                if (stripos($line, $search_term) !== false) {
                    $results[] = [
                        'line' => $line_num + 1,
                        'content' => trim($line)
                    ];
                }
            }
            
            return $results;
        }
        
        throw new Exception("Invalid search syntax");
    }
    
    private function getFromJson($params, $file_path) {
        $data = json_decode(file_get_contents($file_path), true);
        
        if (preg_match("/value\s+from\s+key\s+['\"]([^'\"]+)['\"]/i", $params, $matches)) {
            $key = $matches[1];
            return isset($data[$key]) ? $data[$key] : null;
        }
        
        return $data;
    }
    
    private function getFromCsv($params, $file_path) {
        $data = [];
        $handle = fopen($file_path, 'r');
        
        if ($handle !== false) {
            $headers = fgetcsv($handle);
            
            if ($headers) {
                while (($row = fgetcsv($handle)) !== false) {
                    if (count($row) === count($headers)) {
                        $data[] = array_combine($headers, $row);
                    }
                }
            }
            
            fclose($handle);
        }
        
        return $data;
    }
    
    private function getFromText($params, $file_path) {
        return file_get_contents($file_path);
    }
    
    private function addToCsv($params, $file_path) {
        if (preg_match("/line\s+['\"]([^'\"]+)['\"]/i", $params, $matches)) {
            $line = $matches[1];
            $data = str_getcsv($line);
            
            $handle = fopen($file_path, 'a');
            if ($handle !== false) {
                fputcsv($handle, $data);
                fclose($handle);
                return true;
            }
        }
        
        return false;
    }
    
    private function addToText($params, $file_path) {
        if (preg_match("/line\s+['\"]([^'\"]+)['\"]/i", $params, $matches)) {
            $line = $matches[1];
            return file_put_contents($file_path, $line . "\n", FILE_APPEND);
        }
        
        return false;
    }
    
    private function updateJson($params, $file_path) {
        $data = json_decode(file_get_contents($file_path), true);
        
        if (preg_match("/key\s+['\"]([^'\"]+)['\"]\s+with\s+['\"]([^'\"]+)['\"]/i", $params, $matches)) {
            $key = $matches[1];
            $value = $matches[2];
            
            $data[$key] = $value;
            
            return file_put_contents($file_path, json_encode($data, JSON_PRETTY_PRINT));
        }
        
        return false;
    }
}

?>